%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code:  Example 5.1 
% File: ENSO2.m (F^{(1,3)}_{T} test statistic) 
%
% Programmed by  Bruce E. Hansen.
% For updates and contact information, see the webpage:
% www.ssc.wisc.edu/~bhansen
%
% This program estimates a two-threshold SETAR model, and tests the null 
% of a one-regime AR against the alternative of a three-regime SETAR. 
%
% Reference:
% Hansen, B.E. (1999). Testing for linearity.
%   Journal of Economic Surveys, 13(5), 551-576.
%   DOI: 10.1111/1467-6419.00098.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function ENSO2
yname='ENSO';	                % Name of series 
load ENSOdat.txt;
dat=ENSOdat;  

global p;p=5;		        % Order of autoregression 
global n;
global dmin_;dmin_=1;		% Minimal delay order 
global dmax_;dmax_=p;		% Maximal delay order (set to p or less)
global trim_;trim_ = .1;	% Minimal percentage of data per regime 
global qnum_;qnum_=100;	        % Number of thresholds to search. 
                                % Set qnum_=0 to search over all values 
global boot_;boot_ = 1000;	% Bootstrap replications 
global k;
global t;

% Define data 
n = length(dat(:,1));
y = dat(p+1:n);
t = n-p;
x=[ones(t,1),dat(p:n-1)];
xname  = ['Constant';' Y(t-01)'];
xname2 = ['Constant  ';' Y(t-01)^2'];

j=2;
while j<=p
    x=[x,dat(p+1-j:n-j)];
    if j<10
        kname = strcat('_Y(t-0',num2str(j),')');
    else
        kname = strcat('_Y(t-',num2str(j),')');
    end;
    xname  = [xname;kname];
    xname2 = xname;
    j=j+1;
end;
k=length(x(1,:));

% Linear Regression 
mi   = inv(x'*x);
beta = mi*(x'*y);
e    = y-x*beta;
ee   = e'*e;
xe   = x.*(e*ones(1,length(x(1,:))));
sig  = ee/t;
se   = sqrt(diag(mi*(xe'*xe)*mi));

% Conditional Variance 
e2 = e.*e;
x2 = x.*x;
m2 = inv(x2'*x2);
hetbeta = m2*(x2'*e2);

h   = x2*hetbeta;
eh  = e./sqrt(h.*(h>0)+(h<0)).*(h>0);
u   = e2-h;
x2u = x2.*(u*ones(1,length(x(1,:))));
se2 = sqrt(diag(m2*(x2u'*x2u)*m2));
em  = e2-mean(e2)';
fh  = t*((em'*em)/(u'*u)-1);

% Report Linear Estimation 
fprintf('Dependent Variable:   %s\n',yname);
disp(' ');
disp('Linear Autoregression');
disp(' ');
hname=['Variable     ','Estimate     ','St Error'];
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(beta)
    fprintf('%s     %f    %f\n',xname(i,:),beta(i),se(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',t);
fprintf('Sum of Squared Errors:         %f\n',ee);
fprintf('Residual Variance:             %f\n',sig);
disp(' ');
disp(' ');
disp('Conditional Variance');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(hetbeta)
    fprintf('%s     %f    %f\n',xname2(i,:),hetbeta(i),se2(i));
end;
fprintf('Heteroskedasticity F Test:     %f      %f\n',fh,1-chi2cdf(fh,p));
disp('----------------------------------');
disp(' ');
disp(' ');
disp(' ');
disp(' ');

% SETAR Estimation 
[f,dhat,ghat1,ghat2] = tar2(dat);
q     = qfunc(x);
qd    = q(:,dhat);
delay = dmin_-1+dhat;

d1 = (qd<=ghat1);
d2 = (qd<=ghat2).*(1-d1);
d3 = 1-d1-d2;
x1 = x.*(d1*ones(1,length(x(1,:))));
x2 = x.*(d2*ones(1,length(x(1,:))));
x3 = x.*(d3*ones(1,length(x(1,:))));
xx = [x1,x2,x3];

mxx     = inv(xx'*xx);
betatar = mxx*(xx'*y);
etar    = y-xx*betatar;
xxe     = xx.*(etar*ones(1,length(xx(1,:))));
eetar   = etar'*etar;
sigtar  = eetar/t;

n1 = sum(d1);
n2 = sum(d2);
n3 = sum(d3);
setar = sqrt(diag(mxx*(xxe'*xxe)*mxx));

sig1 = (etar.^2)'*d1/n1;
sig2 = (etar.^2)'*d2/n2;
sig3 = (etar.^2)'*d3/n3;

% Report SETAR estimates 
disp('Threshold Autoregression');
disp(' ');
fprintf('Sum of Squared Errors:         %f\n',eetar);
fprintf('Residual Variance:             %f\n',sigtar);
fprintf('Delay Order:                   %u\n',delay);
fprintf('First Threshold Estimate:      %f\n',ghat1);
fprintf('Second Threshold Estimate:     %f\n',ghat2);
disp(' ');disp(' ');
disp('Regime 1 (Threshold Variable less than or equal to First Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:k
    fprintf('%s     %f    %f\n',xname(i,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n1);
fprintf('Percentage:                    %f\n',n1/t);
fprintf('Regime Variance:              %f\n',sig1);
disp(' ');disp(' ');

disp('Regime 2 (Threshold Variable Between Thresholds):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=k+1:2*k
    fprintf('%s     %f    %f\n',xname(i-k,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n2);
fprintf('Percentage:                    %f\n',n2/t);
fprintf('Regime Variance:               %f\n',sig2);
disp(' ');disp(' ');disp(' ');

disp('Regime 3 (Threshold Variable Above Second Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=(2*k+1):(3*k)
    fprintf('%s     %f    %f\n',xname(i-2*k,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n3);
fprintf('Percentage:                    %f\n',n3/t);
fprintf('Regime Variance:               %f\n',sig3);
disp(' ');disp(' ');disp(' ');

% Model-Based Bootstrap, Homoskedastic Errors   
y0 = dat(1:p);
fb = zeros(boot_,1);
i=1;
while i<=boot_
    [fbi,dhatb,ghat1b,ghat2b] = tar2(ar_sim(y0,beta,e));
    fb(i) = fbi;
    i=i+1;
end;
btpv=mean(fb>(ones(length(fb(:,1)),1)*f))';

% Model-Based Bootstrap, Heteroskedastic Errors 
y0  = dat(1:p);
fhb = zeros(boot_,1);
i=1;
while i<=boot_
    [fbi,dhatb,ghat1b,ghat2b] = tar2(het_sim(y0,beta,hetbeta,eh));
    fbh(i) = fbi;
    i=i+1;
end;
btpvh = mean(fbh>(ones(length(fbh(:,1)),1)*f))';

% Output %
fprintf('Bootstrap Replications:          %u\n',boot_);    
disp(' ');
disp(' ');
disp('TAR Tests, P-Values');
disp(' ');
disp('Fstat      PV-BT     PV-BT-H');
disp('----------------------------');
for i=1:length(btpv)
fprintf('%f   %f   %f\n',f(i),btpv(i),btpvh(i));
end;
disp(' ');
disp(' ');

% Density Calculation 
ub     = max([quant(fb,.99);quant(fbh,.99)])';
k_x    = (0:ub/999:ub)';
k_chi  = (k_x.^(k-1)).*exp(-k_x/2)./(gamma(k)*(2^k));
kern   = [k_chi,kernel(fb,k_x),kernel(fbh,k_x)];
sun_k2 = [k_x,kern];
save('sun_k2.mat','sun_k2');
% TAR Procedures %

function [dhat,ghat]=tar(y,x,q,qq)
mi = ginv(x'*x);
e  = y-x*mi*(x'*y);
qn = length(q(1,:));
gn = length(qq(:,1));
s  = zeros(gn,qn);

m=1;
while m<=qn
    qm = q(:,m);
    j=1;
    while j<=gn
        gg = qq(j,m);
        if isnan(gg)
            continue;
        end;
        xd     = x.*((qm<=gg)*ones(1,length(x(1,:))));
        xxd    = xd'*xd;
        mmi    = xxd-xxd*mi*xxd;
        xde    = xd'*e;
        s(j,m) = xde'*dinv(xde,mmi);
        j=j+1;
    end;
    m=m+1;
end;
[temp,dhat] = max(max(s)');
[temp,kkk]  = max(s(:,dhat));
ghat        = qq(kkk,dhat);
clear temp;
clear kkk;

function [f,ghat]=tar_d(y,x,x1,q,qq)
mi = ginv(x'*x);
e  = y-x*mi*(x'*y);
gn = length(qq(:,1));
s  = zeros(gn,1);

j=1;
while j<=gn;
    gg = qq(j);
    if isnan(gg)
       continue;
    end;
    xd   = [x1,x.*((q<=gg)*ones(1,length(x(1,:))))];
    xxd  = xd'*xd;
    xdx  = x'*xd;
    mmi  = xxd-xdx'*mi*xdx;
    xde  = xd'*e;
    s(j) = xde'*dinv(xde,mmi);
    j=j+1;
end;
[f,temp]=max(s);
ghat = qq(temp);
clear temp;

function qq=qsort(q)
global trim_;
global qnum_;
n  = length(q(:,1));
k  = length(q(1,:));
n1 = round(trim_*n);

if qnum_==0;
    qq=zeros(n-n1-n1+1,k);
    qq=nan*qq;
    j=1;
    while j<=k
        qj = unique(q(:,j),1);
        qj = qj(n1:(length(qj(:,1))-n1+1));
        qq(1:length(qj(:,1)),j) = qj;
        j=j+1;
    end;
else
    qq = zeros(qnum_,k);
    qq = nan*qq;
    j=1;
    while j<=k
        qj = unique(q(:,j));
        qj = qj(n1:(length(qj(:,1))-n1+1));
        nj = length(qj(:,1));
        if nj<=qnum_
            qq(1:nj,j) = qj;
        else
            qq(:,j) = qj(ceil((1:qnum_)./qnum_*nj));
        end;
        j=j+1;
    end;
end;

function qq=qsort_2(q,ghat)
global trim_;
global qnum_;
n  = length(q(:,1));
n1 = round(trim_*n);
qq = unique(q);
qq = qq(n1:(length(qq(:,1))-n1+1));
[temp,kkk] = max(qq>=ghat);
ddd = abs((1:length(qq(:,1)))-kkk)<n1;

temp = 0;
for i=1:length(ddd)
    if ddd(i)==0
        if temp==0
            kkk = qq(i,:);
            temp=1;
        else
            kkk = [kkk;qq(i,:)];
        end;
    end;
end;

qq = kkk;
clear temp;
clear ddd;
clear kkk;
if qnum_>0
    nq=length(qq(:,1));
    if nq>qnum_
        qq = qq(ceil((1:qnum_)'./qnum_*nq));
    end;
end;

function q=qfunc(x)
global dmin_;
global dmax_;
q = x(:,(dmin_+1):(dmax_+1));

function [f,dhat,ghat1,ghat2]=tar2(dat)
global p;
global n;
global t;
y = dat(p+1:n);
x = [ones(t,1),dat(p:n-1)];
j=2;
while j<=p
    x = [x,dat(p+1-j:n-j)];
    j=j+1;
end;
e = y-x*(y'/x')';
q = qfunc(x);
[dhat,ghat1] = tar(y,x,q,qsort(q));                 % fit SETAR-1 model 
qd           = q(:,dhat);
x1           = x.*((qd<=ghat1)*ones(1,length(x(1,:))));
[f,ghat2]    = tar_d(y,x,x1,qd,qsort_2(qd,ghat1));  % fit SETAR-2 model 
x1           = x.*((qd<=ghat2)*ones(1,length(x(1,:))));
[f,ghat1]    = tar_d(y,x,x1,qd,qsort_2(qd,ghat2));  % iterated estimate of gamma_1 %
x1           = x.*((qd<=ghat1)*ones(1,length(x(1,:))));
[f,ghat2]    = tar_d(y,x,x1,qd,qsort_2(qd,ghat1));  % iterated estimate of gamma_2 %
f = t/(((e'*e)/f)-1);
if ghat2<ghat1; 
    gg    = ghat1;
    ghat1 = ghat2;
    ghat2 = gg;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation Procedures 
%%%%%%%%%%%%%%%%%%%%%%%%%
function yb=ar_sim(y0,beta,e)
global k;
t  = length(e(:,1));
u  = e(ceil(unifrnd(0,1,t,1)*t))+beta(1);
yb = y0;
eols_c = [y0;u];
aa     = beta(2:k);

for i=(length(y0(:,1))+1):(length(eols_c(:,1)))
    datbb = eols_c(i,:);
    for j=1:length(y0(:,1))
        datbb = datbb+aa(j,:).*yb(i-j,:);
    end;
    yb = [yb;datbb];
end;

function y=het_sim(y0,beta,het,e)
global p;
t = length(e(:,1));
u = e(ceil(unifrnd(0,1,t,1)*t));
y = [y0;zeros(t,1)];

j=1;
while j<=t
    x=1;
    for i=(p+j-1):-1:j
        x = [x;y(i)];
    end;
    h      = (x.*x)'*het;
    y(p+j) = beta'*x+sqrt(h.*(h>0)).*u(j);
    j=j+1;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function kern=kernel(x,b)
h    = 1.7*std(x)'/(length(x(:,1))^(.2));
g    = length(b(:,1));
kern = zeros(g,1);
i=1;
while i<=g;
    u       = abs(b(i)-x)/h;
    kern(i) = mean((1-u.^2).*(u<=1))'*(.75)/h;
    i=i+1;
end;

function qq=quant(x,q)
s  = sortrows(x,1);
qq = s(round(length(s(:,1))*q));

function mi=ginv(m)
warning off;
lastwarn(' ');
mi = inv(m);
mw=' ';
[mw,idw] = lastwarn;
lastwarn(' ');
warning on;
if mw(1)=='M'
   mi = pinv(m);
end;

function d=dinv(y,x)
warning off;
d = (y'/x')';
[mw,idw] = lastwarn;
lastwarn(' ');
warning on;
if mw(1)=='M'
   d = ginv(x'*x)*(x'*y);
end;
